'use client'

import { useEffect, useState, useRef } from 'react'
import { useParams } from 'next/navigation'
import { supabase } from '@/lib/supabase'
import { Cotizacion, Cliente, CotizacionItem, Producto } from '@/lib/types'
import Link from 'next/link'
import { format } from 'date-fns'
import Image from 'next/image'
import jsPDF from 'jspdf'
import html2canvas from 'html2canvas'

export default function CotizacionDetailPage() {
  const params = useParams()
  const [cotizacion, setCotizacion] = useState<Cotizacion | null>(null)
  const [cliente, setCliente] = useState<Cliente | null>(null)
  const [items, setItems] = useState<(CotizacionItem & { producto?: Producto })[]>([])
  const [loading, setLoading] = useState(true)
  const [generandoPDF, setGenerandoPDF] = useState(false)
  const [mostrandoMensajeWhatsApp, setMostrandoMensajeWhatsApp] = useState(false)
  const cotizacionRef = useRef<HTMLDivElement>(null)

  useEffect(() => {
    if (params.id) {
      fetchCotizacion(params.id as string)
    }
  }, [params.id])

  async function fetchCotizacion(id: string) {
    try {
      const { data: cotizacionData, error: cotizacionError } = await supabase
        .from('cotizaciones')
        .select(`
          *,
          cliente:clientes(*)
        `)
        .eq('id', id)
        .single()

      if (cotizacionError) throw cotizacionError

      setCotizacion(cotizacionData)
      setCliente(cotizacionData.cliente as Cliente)

      // Obtener items de la cotización
      const { data: itemsData, error: itemsError } = await supabase
        .from('cotizaciones_items')
        .select(`
          *,
          producto:productos(*)
        `)
        .eq('cotizacion_id', id)
        .order('orden')

      if (itemsError) throw itemsError
      setItems(itemsData || [])
    } catch (error) {
      console.error('Error fetching cotizacion:', error)
    } finally {
      setLoading(false)
    }
  }

  if (loading) {
    return <div className="text-center p-8">Cargando...</div>
  }

  if (!cotizacion) {
    return (
      <div className="text-center p-8">
        <p>Cotización no encontrada</p>
        <Link href="/cotizaciones" className="text-blue-600 hover:underline">Volver a cotizaciones</Link>
      </div>
    )
  }

  function obtenerColorEstado(estado: string): string {
    switch (estado) {
      case 'aceptada':
        return 'bg-green-100 text-green-800'
      case 'rechazada':
        return 'bg-red-100 text-red-800'
      case 'vencida':
        return 'bg-yellow-100 text-yellow-800'
      default:
        return 'bg-blue-100 text-blue-800'
    }
  }

  async function descargarPDF() {
    if (!cotizacionRef.current || !cotizacion || !cliente) return

    setGenerandoPDF(true)

    try {
      const canvas = await html2canvas(cotizacionRef.current, {
        scale: 2,
        useCORS: true,
        logging: false,
      })

      const imgData = canvas.toDataURL('image/png')
      const pdf = new jsPDF('p', 'mm', 'a4')
      const pdfWidth = pdf.internal.pageSize.getWidth()
      const pdfHeight = pdf.internal.pageSize.getHeight()
      const imgWidth = canvas.width
      const imgHeight = canvas.height
      const ratio = Math.min(pdfWidth / imgWidth, pdfHeight / imgHeight)
      const imgX = (pdfWidth - imgWidth * ratio) / 2
      const imgY = 0

      pdf.addImage(imgData, 'PNG', imgX, imgY, imgWidth * ratio, imgHeight * ratio)
      pdf.save(`cotizacion-${cotizacion.numero_cotizacion}.pdf`)
    } catch (error) {
      console.error('Error generating PDF:', error)
      alert('Error al generar el PDF')
    } finally {
      setGenerandoPDF(false)
    }
  }

  function obtenerURLCotizacion() {
    if (typeof window === 'undefined') return ''
    return `${window.location.origin}/cotizaciones/${cotizacion?.id}`
  }

  function generarMensajeWhatsApp() {
    if (!cotizacion || !cliente) return ''

    const fechaEmision = cotizacion.fecha_emision.split('T')[0]
    const [year, month, day] = fechaEmision.split('-')
    const fechaFormateada = `${day}/${month}/${year}`
    const urlCotizacion = obtenerURLCotizacion()

    const subtotalCalculado = items.reduce((sum, item) => sum + item.subtotal, 0)
    const ivaCalculado = items.reduce((sum, item) => sum + (item.incluir_iva && item.iva ? item.iva : 0), 0)

    let mensaje = `Hola ${cliente.nombre || cliente.empresa || 'Cliente'} 👋

Te envío la cotización ${cotizacion.numero_cotizacion}:

📋 *Detalles de la Cotización:*
• Cliente: ${cliente.nombre || cliente.empresa || 'Cliente'}${cliente.nombre && cliente.empresa ? ` - ${cliente.empresa}` : ''}
• Fecha de emisión: ${fechaFormateada}${cotizacion.fecha_vencimiento ? `\n• Válida hasta: ${format(new Date(cotizacion.fecha_vencimiento), 'dd/MM/yyyy')}` : ''}
• Estado: ${cotizacion.estado === 'pendiente' ? '⏳ Pendiente' : cotizacion.estado === 'aceptada' ? '✅ Aceptada' : cotizacion.estado}

💰 *Resumen de Montos:*
• Subtotal: $${subtotalCalculado.toLocaleString()}${ivaCalculado > 0 ? `\n• IVA (19%): $${ivaCalculado.toLocaleString()}` : ''}
• *Total: $${cotizacion.monto.toLocaleString()}*`

    if (items.length > 0) {
      mensaje += `\n\n📦 *Productos/Servicios:*\n`
      items.forEach((item, index) => {
        mensaje += `${index + 1}. ${item.producto?.nombre || item.descripcion || 'Producto personalizado'}\n`
        mensaje += `   Cantidad: ${item.cantidad} x $${item.precio_unitario.toLocaleString()} = $${item.subtotal.toLocaleString()}`
        if (item.incluir_iva && item.iva) {
          mensaje += ` (IVA: $${item.iva.toLocaleString()})`
        }
        mensaje += `\n`
      })
    }

    mensaje += `\n${cotizacion.descripcion ? `📝 Notas: ${cotizacion.descripcion}\n` : ''}🔗 *Ver cotización completa:*
${urlCotizacion}

Quedo atento a tus comentarios. Si tienes alguna consulta, no dudes en contactarme.

Saludos cordiales,
NAVIGPS`

    return mensaje
  }

  function copiarMensajeWhatsApp() {
    const mensaje = generarMensajeWhatsApp()
    navigator.clipboard.writeText(mensaje).then(() => {
      alert('✅ Mensaje copiado al portapapeles. Ahora puedes pegarlo en WhatsApp.')
    }).catch(() => {
      alert('Error al copiar el mensaje. Por favor, cópialo manualmente.')
    })
  }

  function abrirWhatsApp() {
    const mensaje = generarMensajeWhatsApp()
    const mensajeCodificado = encodeURIComponent(mensaje)
    const telefono = cliente?.telefono?.replace(/\D/g, '') || ''
    
    if (telefono) {
      window.open(`https://wa.me/${telefono}?text=${mensajeCodificado}`, '_blank')
    } else {
      alert('No hay número de teléfono registrado para este cliente.')
    }
  }

  return (
    <div className="max-w-4xl mx-auto">
      <div className="mb-4 flex justify-between items-center">
        <Link href="/cotizaciones" className="text-blue-600 hover:underline">← Volver a cotizaciones</Link>
        <div className="flex gap-2">
          <button
            onClick={descargarPDF}
            disabled={generandoPDF}
            className="bg-green-600 text-white px-4 py-2 rounded hover:bg-green-700 disabled:opacity-50 transition flex items-center gap-2"
          >
            {generandoPDF ? 'Generando...' : '📄 Descargar PDF'}
          </button>
          <button
            onClick={() => setMostrandoMensajeWhatsApp(true)}
            className="bg-green-500 text-white px-4 py-2 rounded hover:bg-green-600 transition flex items-center gap-2"
          >
            💬 WhatsApp
          </button>
        </div>
      </div>

      {/* Modal de WhatsApp */}
      {mostrandoMensajeWhatsApp && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-lg p-6 max-w-2xl w-full max-h-[80vh] overflow-y-auto">
            <div className="flex justify-between items-center mb-4">
              <h2 className="text-2xl font-bold">Mensaje para WhatsApp</h2>
              <button
                onClick={() => setMostrandoMensajeWhatsApp(false)}
                className="text-gray-500 hover:text-gray-700 text-2xl"
              >
                ×
              </button>
            </div>
            <div className="mb-4">
              <textarea
                readOnly
                value={generarMensajeWhatsApp()}
                className="w-full border rounded px-3 py-2 h-64 font-mono text-sm"
              />
            </div>
            <div className="flex gap-2">
              <button
                onClick={copiarMensajeWhatsApp}
                className="flex-1 bg-blue-600 text-white px-4 py-2 rounded hover:bg-blue-700 transition"
              >
                📋 Copiar Mensaje
              </button>
              <button
                onClick={abrirWhatsApp}
                className="flex-1 bg-green-500 text-white px-4 py-2 rounded hover:bg-green-600 transition"
              >
                💬 Abrir WhatsApp
              </button>
              <button
                onClick={() => setMostrandoMensajeWhatsApp(false)}
                className="bg-gray-300 text-gray-700 px-4 py-2 rounded hover:bg-gray-400 transition"
              >
                Cerrar
              </button>
            </div>
          </div>
        </div>
      )}

      <div ref={cotizacionRef} className="bg-white p-6 rounded-lg shadow mb-6">
        {/* Logo y encabezado */}
        <div className="mb-6 pb-4 border-b">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-4">
              <Image
                src="/img/logo_navigps.png"
                alt="NAVIGPS Logo"
                width={120}
                height={60}
                className="object-contain"
              />
              <div>
                <h1 className="text-3xl font-bold">{cotizacion.numero_cotizacion}</h1>
              </div>
            </div>
            <span className={`px-3 py-1 rounded text-sm font-medium ${obtenerColorEstado(cotizacion.estado)}`}>
              {cotizacion.estado}
            </span>
          </div>
        </div>

        {/* Información del Cliente/Empresa */}
        <div className="mb-6 p-4 bg-gray-50 rounded-lg">
          <h3 className="font-semibold text-gray-700 mb-3">Datos del Cliente</h3>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
            <div>
              <span className="text-sm text-gray-600">Nombre:</span>
              <p className="font-semibold text-gray-900">{cliente?.nombre || 'N/A'}</p>
            </div>
            {cliente?.empresa && (
              <div>
                <span className="text-sm text-gray-600">Empresa:</span>
                <p className="font-semibold text-gray-900">{cliente.empresa}</p>
              </div>
            )}
            {cliente?.rfc && (
              <div>
                <span className="text-sm text-gray-600">RUT:</span>
                <p className="font-semibold text-gray-900">{cliente.rfc}</p>
              </div>
            )}
            {cliente?.email && (
              <div>
                <span className="text-sm text-gray-600">Email:</span>
                <p className="font-semibold text-gray-900">{cliente.email}</p>
              </div>
            )}
            {cliente?.telefono && (
              <div>
                <span className="text-sm text-gray-600">Teléfono:</span>
                <p className="font-semibold text-gray-900">{cliente.telefono}</p>
              </div>
            )}
            {cliente?.direccion && (
              <div className="md:col-span-2">
                <span className="text-sm text-gray-600">Dirección:</span>
                <p className="font-semibold text-gray-900">{cliente.direccion}</p>
              </div>
            )}
          </div>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
          <div>
            <h3 className="font-semibold text-gray-700 mb-1">Fecha de Emisión</h3>
            <p className="text-gray-900">
              {format(new Date(cotizacion.fecha_emision), 'dd/MM/yyyy')}
            </p>
          </div>
          {cotizacion.fecha_vencimiento && (
            <div>
              <h3 className="font-semibold text-gray-700 mb-1">Fecha de Vencimiento</h3>
              <p className="text-gray-900">
                {format(new Date(cotizacion.fecha_vencimiento), 'dd/MM/yyyy')}
              </p>
            </div>
          )}
          <div>
            <h3 className="font-semibold text-gray-700 mb-1">Monto Total</h3>
            <p className="text-2xl font-bold text-gray-900">${cotizacion.monto.toLocaleString()}</p>
          </div>
        </div>

        {/* Desglose de IVA si aplica */}
        {cotizacion.incluir_iva && cotizacion.subtotal && cotizacion.iva && (
          <div className="mb-6 p-4 bg-gray-50 rounded-lg">
            <h3 className="font-semibold text-gray-700 mb-3">Desglose de Montos</h3>
            <div className="space-y-2">
              <div className="flex justify-between">
                <span className="text-gray-600">Subtotal:</span>
                <span className="font-semibold">${cotizacion.subtotal.toLocaleString()}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">IVA (19%):</span>
                <span className="font-semibold">${cotizacion.iva.toLocaleString()}</span>
              </div>
              <div className="flex justify-between border-t pt-2">
                <span className="font-bold">Total:</span>
                <span className="font-bold text-lg">${cotizacion.monto.toLocaleString()}</span>
              </div>
            </div>
          </div>
        )}

        {cotizacion.descripcion && (
          <div className="mb-6">
            <h3 className="font-semibold text-gray-700 mb-2">Descripción</h3>
            <p className="text-gray-900">{cotizacion.descripcion}</p>
          </div>
        )}

        {/* Items de la cotización */}
        <div className="mb-6">
          <h3 className="text-xl font-bold mb-4">Productos y Servicios</h3>
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-4 py-3 text-left text-sm font-semibold">Producto/Servicio</th>
                  <th className="px-4 py-3 text-left text-sm font-semibold">Cantidad</th>
                  <th className="px-4 py-3 text-left text-sm font-semibold">Precio Unit.</th>
                  <th className="px-4 py-3 text-left text-sm font-semibold">Subtotal</th>
                  <th className="px-4 py-3 text-left text-sm font-semibold">IVA</th>
                </tr>
              </thead>
              <tbody>
                {items.map((item) => (
                  <tr key={item.id} className="border-t">
                    <td className="px-4 py-3">
                      <div>
                        <p className="font-medium">
                          {item.producto?.nombre || item.descripcion || 'Producto personalizado'}
                        </p>
                        {item.descripcion && item.producto && (
                          <p className="text-sm text-gray-500">{item.descripcion}</p>
                        )}
                      </div>
                    </td>
                    <td className="px-4 py-3">{item.cantidad}</td>
                    <td className="px-4 py-3">${item.precio_unitario.toLocaleString()}</td>
                    <td className="px-4 py-3 font-semibold">${item.subtotal.toLocaleString()}</td>
                    <td className="px-4 py-3">
                      {item.incluir_iva && item.iva ? (
                        <span className="text-green-600 font-semibold">${item.iva.toLocaleString()}</span>
                      ) : (
                        <span className="text-gray-400">-</span>
                      )}
                    </td>
                  </tr>
                ))}
              </tbody>
              <tfoot className="bg-gray-50">
                {(() => {
                  const subtotalCalculado = items.reduce((sum, item) => sum + item.subtotal, 0)
                  const ivaCalculado = items.reduce((sum, item) => sum + (item.incluir_iva && item.iva ? item.iva : 0), 0)
                  const tieneIva = ivaCalculado > 0
                  
                  return tieneIva ? (
                    <>
                      <tr>
                        <td colSpan={4} className="px-4 py-3 text-right">Subtotal:</td>
                        <td className="px-4 py-3">${subtotalCalculado.toLocaleString()}</td>
                      </tr>
                      <tr>
                        <td colSpan={4} className="px-4 py-3 text-right">IVA (19%):</td>
                        <td className="px-4 py-3">${ivaCalculado.toLocaleString()}</td>
                      </tr>
                      <tr>
                        <td colSpan={4} className="px-4 py-3 text-right font-bold">Total:</td>
                        <td className="px-4 py-3 font-bold text-lg">${cotizacion.monto.toLocaleString()}</td>
                      </tr>
                    </>
                  ) : (
                    <tr>
                      <td colSpan={4} className="px-4 py-3 text-right font-bold">Total:</td>
                      <td className="px-4 py-3 font-bold text-lg">${cotizacion.monto.toLocaleString()}</td>
                    </tr>
                  )
                })()}
              </tfoot>
            </table>
          </div>
        </div>

        {/* Pie de página */}
        <div className="mt-8 pt-6 border-t border-gray-300">
          <div className="text-center text-sm text-gray-600">
            <p className="font-semibold mb-1">NAVIGPS</p>
            <p><a href="mailto:info@navigps.cl" className="text-blue-600 hover:underline">info@navigps.cl</a></p>
            <p><a href="https://navigps.cl" target="_blank" rel="noopener noreferrer" className="text-blue-600 hover:underline">navigps.cl</a></p>
          </div>
        </div>
      </div>
    </div>
  )
}
